%This code accompanies: 
%
%Prins, N. (2012). The Psychometric Function: The Lapse Rate Revisited. 
%Journal of Vision.
%
%(henceforth referred to as LRR).
%
%This routine performs simulations and fits for any of the conditions shown
%in Figure 9 of LRR. For each simulation it will show a figure akin to
%Figure 3 in LRR. At completion it will show a Figure akin to Figure 7 and
%a Figure akin to Figure 5.
%
%Developed under Matlab R2012a.
%
%This routine calls on routines in the Palamedes Toolbox version 1.5.0.
%Only the routines in the Palamedes toolbox necessary to perform these
%replications were provided with this script.
%
%Full toolbox available at www.palamedestoolbox.org
%
%Prins, N. & Kingdom, F.A.A. (2009). Palamedes: Matlab routines for 
%analyzing psychophysical data. www.palamedestoolbox.org.
%
%NP, June 2012

clear all

s = RandStream.create('mt19937ar','seed',sum(100*clock));
RandStream.setGlobalStream(s);

warningstates = warning('query','all');%avoids inconsequential Log of Zero
warning off all                        %warnings

Ntrial = 0;                                        
while Ntrial ~= 120 && Ntrial ~= 240 && Ntrial ~= 480 && Ntrial ~= 960
    Ntrial = input('N (120, 240, 480, or 960): ');
end
disp('Simulations are computationally very intensive! Try low number of simulations first');
numSims = input('Number of simulations (e.g., 2000): ');
lapseGen = input('Generating lapse rate (0 through 0.05): ');

stepThru = strcmpi(input('Step through by spacebar? (y/n): ','s'),'y');

switch Ntrial %Vary axis limit etc. with N
    case 120
        crit = log10([5 17; 3.5 22; .002 9; .0005 16]);        
    case 240
        crit = log10([6 14; 5 16; .008 2; .003 3]);        
    case 480
        crit = log10([7 12; 6 13; .02 1; .008 1.4]);        
    case 960
        crit = log10([7 11; 6.5 12; .03 .5; .02 .6]);        
end
%options for simplex search
options = PAL_minimize('options');
options.TolFun = 1e-09;     %low tolerance (i.e., high precision)
options.TolX = 1e-09;       %low tolerance (i.e., high precision)
options.MaxFunEvals = 10000;
options.MaxIter = 10000;
options.Display = 'off';

PF = @PAL_Weibull;      %Generating and fitted function

lapseLimits = [0 0.06];                 %prior on lambda used in nAPLE

lapseIntensity = PF([10 3],.9999,'inverse');    %Stimulus intensity at which F is effectively unity

paramsFit = zeros(numSims,6,4);           %pre-allocate memory
paramsFitAdj = zeros(numSims,6,2);
LL = zeros(numSims,6);
exitflag = zeros(numSims,6);

%%%%%%%%%%%%PSI method parameters

priorAlphaRange = logspace(log10(PF([10 3 0 0],.1,'inverse')),log10(PF([10 3 0 0],.9,'inverse')),51); %values of alpha to include in prior
priorBetaRange = 0:(1- 0)/40:1; %values of log10(beta) to include in prior

gamma = 0.5;            %Guess rate to be assumed
lambda = .025;          %Lapse Rate to be assumed

%Stimulus values to select from (note that 'lapseIntensity' will actually not
%be used during psi-method run, see call to PAL_AMPM_setupPM below)
stimRange = [logspace(log10(PF([10 3 0 0],.1,'inverse')),log10(PF([10 3 0 0],.999,'inverse')),21) lapseIntensity];   

%Prior is uniform within constrained parameter space:
prior = ones(length(priorAlphaRange), length(priorBetaRange));
prior = prior./sum(sum(prior)); %prior should sum to 1

%for plotting later
xFine = PF([10 3 0 0],.05,'inverse'):(PF([10 3 0 0],.9999,'inverse')-PF([10 3 0 0],.05,'inverse'))/100:PF([10 3 0 0],.9999,'inverse');

%Create 3D parameter space (threshold, slope, lapse) and log probability 
%look-up-tables to be used during brute-force search for seeds for simplex 
%search.
%Note: this could be performed within function PAL_PFML_Fit (or
%PAL_PFML_BruteForceFit) but is moved outside of simulation loop to reduce
%computational load.

gridGrain = 150;

alphas = logspace(log10(PF([10 3 0 0],.01,'inverse')),log10(PF([10 3 0 0],.99,'inverse')),gridGrain);    %values of alpha to include in grid
betas = logspace(-1,2,gridGrain); %values of beta to include in grid
lambdas = 0:0.01:0.3;       %values of lambda to include in grid

[params.alpha params.beta params.lambda] = ndgrid(alphas,betas,lambdas);
params.gamma = ones(size(params.alpha)).*0.5;

logpcorrect = zeros(size(stimRange,2),size(params.alpha,1),size(params.alpha,2),size(params.alpha,3));
logpincorrect = zeros(size(stimRange,2),size(params.alpha,1),size(params.alpha,2),size(params.alpha,3));
for level = 1:size(stimRange,2)
    logpcorrect(level,:,:,:) = log(PF(params,stimRange(1,level)));
    logpincorrect(level,:,:,:) = log(1-PF(params,stimRange(1,level)));
end

paramsGen = [10 3 .5 lapseGen]; %generating parameters

figure('units','pixels','position',[10 10 1000 700]);
[gridx gridy] = meshgrid(1:gridGrain,1:gridGrain);

%Simulation loop
for simulation = 1:numSims    
    if mod(simulation,10) == 0    %for the impatient
        disp(int2str(simulation));
    end

    %set up psi-method. Only 5/6 of trials performed using psi-method
    PM = PAL_AMPM_setupPM('priorAlphaRange',priorAlphaRange,...
        'priorBetaRange',priorBetaRange, 'numtrials',Ntrial*5/6, 'PF' , PF,...
        'prior',prior,'stimRange',stimRange(1:length(stimRange)-1),'gamma',gamma,'lambda',lambda);
        
    %trial loop
    while PM.stop ~= 1

        response = rand(1) < PF(paramsGen, PM.xCurrent);
        PM = PAL_AMPM_updatePM(PM,response);

    end

    StimLevelsTemp = PM.x(1:Ntrial*5/6);
    NumPosTemp = PM.response;

    %group trials of equal stimulus intensity
    [StimLevels NumPos OutOfNum] = PAL_PFML_GroupTrialsbyX(StimLevelsTemp, NumPosTemp, ones(1, Ntrial*5/6));
    
    %simulate observer at asymptotic intensity
    NumPosLapse = PAL_PF_SimulateObserverParametric(paramsGen, lapseIntensity*ones(1,Ntrial/6), ones(1,Ntrial/6), PF);

    %combine data
    StimLevels(length(StimLevels)+1) = lapseIntensity;
    NumPos(length(NumPos)+1) = sum(NumPosLapse);
    OutOfNum(length(OutOfNum)+1) = Ntrial/6;
    
%%%%%%%%%%%%iAPLE fit
    
    %First perform brute force search for initial values to seed simplex
    %search with. This is moved outside of function PAL_PFML_Fit to reduce 
    %computational load (see above).

    %Estimate lapse rate in isolation using highest intensity only.
    
    lapseEst = 1-(NumPos(length(NumPos))/OutOfNum(length(OutOfNum))); %lapse rate estimate
    lapseIndex = min(round(lapseEst*100) + 1,31);   %nearest lapse rate in brute-force grid

    %Create likelihood grid and find maximum in it.
    %Only use stimulus intensities other than 'lapseIntensity'.    
    LLspace = zeros(size(params.alpha,1),size(params.alpha,2));
    for level = 1:length(OutOfNum)-1
       LLspace = LLspace + NumPos(level).*squeeze(logpcorrect(find(stimRange == StimLevels(level)),:,:,lapseIndex))+(OutOfNum(level)-NumPos(level)).*squeeze(logpincorrect(find(stimRange == StimLevels(level)),:,:,lapseIndex));
    end
    
    [trash I] = PAL_findMax(LLspace);        
    paramsInit.alpha = alphas(I(1));
    paramsInit.beta = betas(I(2));
    paramsInit.gamma = .5;
    paramsInit.lambda = lapseEst;
    
    %fit function using simplex:
    paramsFree = [1 1 0 1];
    [paramsFit(simulation,3,:) LL(simulation,3) exitflag(simulation,3)] = PAL_PFML_Fit(StimLevels(1:length(StimLevels)), NumPos(1:length(StimLevels)), OutOfNum(1:length(StimLevels)), paramsInit, paramsFree, PF,'searchoptions',options,'lapseFit','iAPLE','lapseLimits',[0 1]);

    %transform to metric reported by Wichmann & Hill
    paramsFitAdj(simulation,3,1) = PF([paramsFit(simulation,3,1) paramsFit(simulation,3,2) 0 0],.5,'inverse');
    paramsFitAdj(simulation,3,2) = PF([paramsFit(simulation,3,1) paramsFit(simulation,3,2) 0 0],paramsFitAdj(simulation,3,1),'derivative');

%%%%%%%%%%%%%%%jAPLE fit

    LLspace = zeros(size(params.alpha,1),size(params.alpha,2),size(params.alpha,3));

    %proportion correct at highest intensity modeled as 1 - lambda:
    LLspace = LLspace + log((1-params.lambda).^NumPos(length(OutOfNum)))+log(params.lambda.^(OutOfNum(length(OutOfNum))-NumPos(length(OutOfNum))));
    %others as usual:
    for level = 1:length(OutOfNum)-1
        LLspace = LLspace + NumPos(level).*squeeze(logpcorrect(find(stimRange == StimLevels(level)),:,:,:))+(OutOfNum(level)-NumPos(level)).*squeeze(logpincorrect(find(stimRange == StimLevels(level)),:,:,:));
    end    
    
    [trash I] = PAL_findMax(LLspace);
    paramsInit.alpha = alphas(I(1));
    paramsInit.beta = betas(I(2));
    paramsInit.gamma = .5;
    paramsInit.lambda = lambdas(I(3));    
    
    %fit function using simplex
    paramsFree = [1 1 0 1];
    [paramsFit(simulation,2,:) LL(simulation,2) exitflag(simulation,2)] = PAL_PFML_Fit(StimLevels, NumPos, OutOfNum, paramsInit, paramsFree, PF,'searchoptions',options,'lapseFit','jAPLE','lapseLimits',[0 1]);

    %transform to metric reported by Wichmann & Hill
    paramsFitAdj(simulation,2,1) = PF([paramsFit(simulation,2,1) paramsFit(simulation,2,2) 0 0],.5,'inverse');
    paramsFitAdj(simulation,2,2) = PF([paramsFit(simulation,2,1) paramsFit(simulation,2,2) 0 0],paramsFitAdj(simulation,2,1),'derivative');
    
    clf('reset')    
    axes('units','pixels','position',[60 400 250 250]);
    hold on
        
    [LLspace2D Ilambda] = max(LLspace,[],3); %collapse to 2D (alpha x beta) and note best lambdas    
    maximLL = max(max(LLspace2D));
    LLspace2D = exp(LLspace2D-maximLL);
    contoursat = exp(-1*[.5 2 4.5 8]);  %contours at 1, 2, 3, and 4 SEs (see: numerical recipes in C)
                                        %Thanks to Stan Klein for this idea.

    cm = flipud(colormap('jet'))*(16/20) + 1-(16/20);
    imagesc(Ilambda),colormap(cm);
    caxis([1 31])    
    plot(I(2),I(1),'k+','markersize',12,'linewidth',2)
    contour(gridx,gridy,LLspace2D,contoursat,'linewidth',1.5,'color','k');
    xlabel('Slope (\beta)','fontsize',14);
    ylabel('Threshold (\alpha)','fontsize',14);
    axis([1 gridGrain 1 gridGrain]);

    line([I(2)-15 I(2)-15],[I(1)-15 I(1)+15],'color','k','linewidth',1.5)
    line([I(2)+15 I(2)+15],[I(1)-15 I(1)+15],'color','k','linewidth',1.5)
    line([I(2)-15 I(2)+15],[I(1)-15 I(1)-15],'color','k','linewidth',1.5)
    line([I(2)-15 I(2)+15],[I(1)+15 I(1)+15],'color','k','linewidth',1.5)
    
    S = [1 10 100];
    xtick = [1 ((log10(S)-log10(.1)).*((gridGrain-1)/(log10(100)-log10(.1))))];
    set(gca,'xtick',xtick,'xticklabel',{'.1','1','10','100'})

    T = [2.2 4 6 10 16];
    ytick = ((log10(T)-log10(PF([10 3 0 0],.01,'inverse'))).*((gridGrain-1)/(log10(PF([10 3 0 0],.99,'inverse'))-log10(PF([10 3 0 0],.01,'inverse')))));
    set(gca,'ytick',ytick,'yticklabel',{'2.2','4','6','10','16'})

    text(1,160,'jAPLE:','fontsize',14);

    set(gca,'fontsize',13);    
    set(gca,'handlevisibility','off');

    %%%%%%%%%%%%%Close up

    minAlphaI = double(max(1,I(1)-15));
    maxAlphaI = double(min(I(1)+15,gridGrain));
    minBetaI = double(max(1,I(2)-15));
    maxBetaI = double(min(I(2)+15,gridGrain));
    
    LLspaceCloseUp = LLspace2D(minAlphaI:maxAlphaI,minBetaI:maxBetaI);
    surf(minBetaI:maxBetaI,minAlphaI:maxAlphaI,LLspaceCloseUp,Ilambda(minAlphaI:maxAlphaI,minBetaI:maxBetaI))
    set(gca,'units','pixels','position',[375 390 250 250]);
    xtick = [.1 .13 .17 .2 .25 .35 .5 .75 1 1.5 2 3 4 5 7 10 15 20 30 40 50 70 100];
    for tick = 1:length(xtick)
        xticklabel(tick) = cellstr(num2str(xtick(tick)));
    end
    xtick = (((log10(xtick)+1)*149/3)+1);
    set(gca,'xtick',xtick,'xticklabel',xticklabel)

    ytick = [2.2 2.5 2.8 3.1 3.5 4 5 6 7 8 9 10 11 12 13 14 15 16];
    for tick = 1:length(ytick)
        yticklabel(tick) = cellstr(num2str(ytick(tick)));
    end
    ytick = (((log10(ytick)-log10(alphas(1)))*149/  (log10(alphas(gridGrain))-log10(alphas(1))))   +1);
    set(gca,'ytick',ytick,'yticklabel',yticklabel)

    set(gca,'ztick',[]);
    set(gca,'zgrid','off','xgrid','off','ygrid','off');
    
    zlabel('Likelihood','fontsize',13);

    axis([minBetaI maxBetaI minAlphaI maxAlphaI 0 1]);    
    caxis([1 31])
    text(double(minBetaI-(maxBetaI-minBetaI)/3.25),mean([minAlphaI maxAlphaI]),'Threshold (\alpha)','rotation',-29,'fontsize',12,'horizontalalignment','center','fontsize',12);
    text(mean([minBetaI maxBetaI]),double(minAlphaI-(maxAlphaI-minAlphaI)/3.25),'Slope (\beta)','rotation',14,'fontsize',12,'horizontalalignment','center','fontsize',12);
    set(gca,'fontsize',13);

    set(gca,'handlevisibility','off');

    %%%%%%%%%%%%colorbar
    cb = repmat(1:32,[2 1]);
    pcolor(cb)
    set(gca,'xgrid','off','ygrid','off');
    set(gca,'xtick',[],'ytick',[]);
    for i = 0:6
        Tl = num2str(i/20,'%.2f');
        text((i*5)+1.5,2.5,Tl,'rotation',45);
    end
    text(16,0,'Lapse rate','horizontalalignment','center','fontsize',12)
    set(gca,'units','pixels','position',[375 630 250 20]);
    set(gca,'handlevisibility','off');
    
    %%%%%%%%PFs

    paramsFitFree = squeeze(paramsFit(simulation,2,:));

    semilogx(xFine,PF(paramsGen,xFine),'-','color',[0 0 0],'linewidth',2)
    hold on
    
    semilogx(xFine,PF(paramsFitFree,xFine),'-','color',[1 0 0],'linewidth',2)

    axis([3 23 .4 1.02])   
    for level = 1:length(StimLevels)
        semilogx(StimLevels(level),NumPos(level)./OutOfNum(level),'o','color',[0 0 0],'markersize',8,'markerfacecolor','k','markersize',16*sqrt(OutOfNum(level)/max(OutOfNum)))
    end
    ylabel('Proportion Correct','fontsize',14);
    xlabel('Stimulus Intensity','fontsize',14);
    set(gca,'xtick',[4:23],'xticklabel',{'4','5','6','7','8','9','','11','','13','','','16','','','','','21','',''});
    set(gca,'ytick',[.5:.1:1]);
    set(gca,'units','pixels','position',[700 400 250 250])
    set(gca,'fontsize',13);

    set(gca,'handlevisibility','off')

    semilogx(xFine,PF([10 3 0 0],xFine),'-','color',[0 0 0],'linewidth',1)
    axis([3 23 0 1])
    hold on
    semilogx(xFine,PF([paramsFitFree(1) paramsFitFree(2) 0 0],xFine),'-','color',[1 0 0],'linewidth',1)
    set(gca,'units','pixels','position',[720 555 75 75])
    set(gca,'ytick',[0:1]);
    set(gca,'xtick',[4:23],'xticklabel',{'4','','','','','','','','','','','','','','','','','21','',''});
    set(gca,'fontsize',13);
    text(2.5,.5,'F(\itx\rm)','horizontalalignment','center','rotation',90,'fontsize',13)    
    set(gca,'handlevisibility','off')    
    

    
%%%%%%%%%%%%%%%nAPLE fit

    %Brute-force search
    LLspace = zeros(size(params.alpha,1),size(params.alpha,2),7);
    for level = 1:size(OutOfNum,2)
       LLspace = LLspace + NumPos(level).*squeeze(logpcorrect(find(stimRange == StimLevels(level)),:,:,1:7))+(OutOfNum(level)-NumPos(level)).*squeeze(logpincorrect(find(stimRange == StimLevels(level)),:,:,1:7));
    end
    
    [trash I] = PAL_findMax(LLspace(:,:,1:7));
    paramsInit.alpha = alphas(I(1));
    paramsInit.beta = betas(I(2));
    paramsInit.gamma = .5;
    paramsInit.lambda = lambdas(I(3));

    %fit function using simplex
    paramsFree = [1 1 0 1];
    [paramsFit(simulation,1,:) LL(simulation,1) exitflag(simulation,1)] = PAL_PFML_Fit(StimLevels, NumPos, OutOfNum, paramsInit, paramsFree, PF,'lapseLimits',lapseLimits,'searchoptions',options);

    %transform to metric reported by Wichmann & Hill
    paramsFitAdj(simulation,1,1) = PF([paramsFit(simulation,1,1) paramsFit(simulation,1,2) 0 0],.5,'inverse');
    paramsFitAdj(simulation,1,2) = PF([paramsFit(simulation,1,1) paramsFit(simulation,1,2) 0 0],paramsFitAdj(simulation,1,1),'derivative');

    axes('units','pixels','position',[60 50 250 250]);
    hold on
        
    [LLspace2D Ilambda] = max(LLspace(:,:,1:7),[],3); %collapse to 2D (alpha x beta) and note best lambdas    
    maximLL = max(max(LLspace2D));
    LLspace2D = exp(LLspace2D-maximLL);
    contoursat = exp(-1*[.5 2 4.5 8]);  %contours at 1, 2, 3, and 4 SEs (see: numerical recipes in C)
                                        %Thanks to Stan Klein for this idea.

    cm = flipud(colormap('jet'))*(16/20) + 1-(16/20);
    imagesc(Ilambda),colormap(cm);
    caxis([1 31])    
    plot(I(2),I(1),'k+','markersize',12,'linewidth',2)
    contour(gridx,gridy,LLspace2D,contoursat,'linewidth',1.5,'color','k');
    xlabel('Slope (\beta)','fontsize',14);
    ylabel('Threshold (\alpha)','fontsize',14);
    axis([1 gridGrain 1 gridGrain]);

    line([I(2)-15 I(2)-15],[I(1)-15 I(1)+15],'color','k','linewidth',1.5)
    line([I(2)+15 I(2)+15],[I(1)-15 I(1)+15],'color','k','linewidth',1.5)
    line([I(2)-15 I(2)+15],[I(1)-15 I(1)-15],'color','k','linewidth',1.5)
    line([I(2)-15 I(2)+15],[I(1)+15 I(1)+15],'color','k','linewidth',1.5)
    
    S = [1 10 100];
    xtick = [1 ((log10(S)-log10(.1)).*((gridGrain-1)/(log10(100)-log10(.1))))];
    set(gca,'xtick',xtick,'xticklabel',{'.1','1','10','100'})

    T = [2.2 4 6 10 16];
    ytick = ((log10(T)-log10(PF([10 3 0 0],.01,'inverse'))).*((gridGrain-1)/(log10(PF([10 3 0 0],.99,'inverse'))-log10(PF([10 3 0 0],.01,'inverse')))));
    set(gca,'ytick',ytick,'yticklabel',{'2.2','4','6','10','16'})

    text(1,160,'nAPLE:','fontsize',14);    
    
    set(gca,'fontsize',13);    
    set(gca,'handlevisibility','off');

    %%%%%%%%%%%%%Close up

    minAlphaI = double(max(1,I(1)-15));
    maxAlphaI = double(min(I(1)+15,gridGrain));
    minBetaI = double(max(1,I(2)-15));
    maxBetaI = double(min(I(2)+15,gridGrain));
    
    LLspaceCloseUp = LLspace2D(minAlphaI:maxAlphaI,minBetaI:maxBetaI);
    surf(minBetaI:maxBetaI,minAlphaI:maxAlphaI,LLspaceCloseUp,Ilambda(minAlphaI:maxAlphaI,minBetaI:maxBetaI))
    set(gca,'units','pixels','position',[375 40 250 250]);
    xtick = [.1 .13 .17 .2 .25 .35 .5 .75 1 1.5 2 3 4 5 7 10 15 20 30 40 50 70 100];
    for tick = 1:length(xtick)
        xticklabel(tick) = cellstr(num2str(xtick(tick)));
    end
    xtick = (((log10(xtick)+1)*149/3)+1);
    set(gca,'xtick',xtick,'xticklabel',xticklabel)

    ytick = [2.2 2.5 2.8 3.1 3.5 4 5 6 7 8 9 10 11 12 13 14 15 16];
    for tick = 1:length(ytick)
        yticklabel(tick) = cellstr(num2str(ytick(tick)));
    end
    ytick = (((log10(ytick)-log10(alphas(1)))*149/  (log10(alphas(gridGrain))-log10(alphas(1))))   +1);
    set(gca,'ytick',ytick,'yticklabel',yticklabel)

    set(gca,'ztick',[]);
    set(gca,'zgrid','off','xgrid','off','ygrid','off');
    
    zlabel('Likelihood','fontsize',13);

    axis([minBetaI maxBetaI minAlphaI maxAlphaI 0 1]);    
    caxis([1 31])
    text(double(minBetaI-(maxBetaI-minBetaI)/3.25),mean([minAlphaI maxAlphaI]),'Threshold (\alpha)','rotation',-29,'fontsize',12,'horizontalalignment','center','fontsize',12);
    text(mean([minBetaI maxBetaI]),double(minAlphaI-(maxAlphaI-minAlphaI)/3.25),'Slope (\beta)','rotation',14,'fontsize',12,'horizontalalignment','center','fontsize',12);
    set(gca,'fontsize',13);

    set(gca,'handlevisibility','off');

    %%%%%%%%%%%%colorbar
    cb = repmat(1:8,[2 1]);
    pcolor(cb);
    caxis([1 31]);
    set(gca,'xgrid','off','ygrid','off');
    set(gca,'xtick',[],'ytick',[]);
    for i = 0:6
        Tl = num2str(i/100,'%.2f');
        text(i+1.5,2.5,Tl,'rotation',45);
    end
    text(4.5,0,'Lapse rate','horizontalalignment','center','fontsize',12)
    set(gca,'units','pixels','position',[375 280 250 20]);
    set(gca,'handlevisibility','off');
    
    %%%%%%%%PFs

    paramsFitFree = squeeze(paramsFit(simulation,1,:));

    semilogx(xFine,PF(paramsGen,xFine),'-','color',[0 0 0],'linewidth',2)
    hold on
    
    semilogx(xFine,PF(paramsFitFree,xFine),'-','color',[1 0 0],'linewidth',2)

    axis([3 23 .4 1.02])   
    for level = 1:length(StimLevels)
        semilogx(StimLevels(level),NumPos(level)./OutOfNum(level),'o','color',[0 0 0],'markersize',8,'markerfacecolor','k','markersize',16*sqrt(OutOfNum(level)/max(OutOfNum)))
    end
    ylabel('Proportion Correct','fontsize',14);
    xlabel('Stimulus Intensity','fontsize',14);
    set(gca,'xtick',[4:23],'xticklabel',{'4','5','6','7','8','9','','11','','13','','','16','','','','','21','',''});
    set(gca,'ytick',[.5:.1:1]);
    set(gca,'units','pixels','position',[700 50 250 250])
    set(gca,'fontsize',13);

    set(gca,'handlevisibility','off')

    semilogx(xFine,PF([10 3 0 0],xFine),'-','color',[0 0 0],'linewidth',1)
    axis([3 23 0 1])
    hold on
    semilogx(xFine,PF([paramsFitFree(1) paramsFitFree(2) 0 0],xFine),'-','color',[1 0 0],'linewidth',1)
    set(gca,'units','pixels','position',[720 205 75 75])
    set(gca,'ytick',[0:1]);
    set(gca,'xtick',[4:23],'xticklabel',{'4','','','','','','','','','','','','','','','','','21','',''});
    set(gca,'fontsize',13);
    text(2.5,.5,'F(\itx\rm)','horizontalalignment','center','rotation',90,'fontsize',13)    
    set(gca,'handlevisibility','off')    
    
    drawnow
    
%%%%%%%Retest as MOCS using placement generated by Psi Method (i.e., now response and placement are independent):

    NumPos = PAL_PF_SimulateObserverParametric(paramsGen, StimLevels, OutOfNum, PF);

%%%%%%%%%%%%iAPLE fit
    
    lapseEst = 1-(NumPos(length(NumPos))/OutOfNum(length(OutOfNum))); %lapse rate estimate
    lapseIndex = min(round(lapseEst*100) + 1,31);   %nearest lapse rate in brute-force grid

    LLspace = zeros(size(params.alpha,1),size(params.alpha,2));
    for level = 1:length(OutOfNum)-1
       LLspace = LLspace + NumPos(level).*squeeze(logpcorrect(find(stimRange == StimLevels(level)),:,:,lapseIndex))+(OutOfNum(level)-NumPos(level)).*squeeze(logpincorrect(find(stimRange == StimLevels(level)),:,:,lapseIndex));
    end
    
    [trash I] = PAL_findMax(LLspace);    
    paramsInit.alpha = alphas(I(1));
    paramsInit.beta = betas(I(2));
    paramsInit.gamma = .5;
    paramsInit.lambda = lapseEst;
    
    paramsFree = [1 1 0 1];
    [paramsFit(simulation,6,:) LL(simulation,6) exitflag(simulation,6)] = PAL_PFML_Fit(StimLevels(1:length(StimLevels)), NumPos(1:length(StimLevels)), OutOfNum(1:length(StimLevels)), paramsInit, paramsFree, PF,'searchoptions',options,'lapseFit','iAPLE','lapseLimits',[0 1]);

    %transform to metric reported by Wichmann & Hill
    paramsFitAdj(simulation,6,1) = PF([paramsFit(simulation,6,1) paramsFit(simulation,6,2) 0 0],.5,'inverse');
    paramsFitAdj(simulation,6,2) = PF([paramsFit(simulation,6,1) paramsFit(simulation,6,2) 0 0],paramsFitAdj(simulation,6,1),'derivative');
    
%%%%%%%%%%%%%%%jAPLE fit

    LLspace = zeros(size(params.alpha,1),size(params.alpha,2),size(params.alpha,3));
    LLspace = LLspace + log((1-params.lambda).^NumPos(length(OutOfNum)))+log(params.lambda.^(OutOfNum(length(OutOfNum))-NumPos(length(OutOfNum))));
    for level = 1:length(OutOfNum)-1
        LLspace = LLspace + NumPos(level).*squeeze(logpcorrect(find(stimRange == StimLevels(level)),:,:,:))+(OutOfNum(level)-NumPos(level)).*squeeze(logpincorrect(find(stimRange == StimLevels(level)),:,:,:));
    end
    
    [trash I] = PAL_findMax(LLspace);
    paramsInit.alpha = alphas(I(1));
    paramsInit.beta = betas(I(2));
    paramsInit.gamma = .5;
    paramsInit.lambda = lambdas(I(3));

    paramsFree = [1 1 0 1];
    [paramsFit(simulation,5,:) LL(simulation,5) exitflag(simulation,5)] = PAL_PFML_Fit(StimLevels(1:length(StimLevels)), NumPos(1:length(StimLevels)), OutOfNum(1:length(StimLevels)), paramsInit, paramsFree, PF,'searchoptions',options,'lapseFit','jAPLE','lapseLimits',[0 1]);
    
    %transform to metric reported by Wichmann & Hill
    paramsFitAdj(simulation,5,1) = PF([paramsFit(simulation,5,1) paramsFit(simulation,5,2) 0 0],.5,'inverse');
    paramsFitAdj(simulation,5,2) = PF([paramsFit(simulation,5,1) paramsFit(simulation,5,2) 0 0],paramsFitAdj(simulation,5,1),'derivative');
   
%%%%%%%%%%%%%%%nAPLE fit

    LLspace = zeros(size(params.alpha,1),size(params.alpha,2),7);%size(params.alpha,3));
    for level = 1:size(OutOfNum,2)
       LLspace = LLspace + NumPos(level).*squeeze(logpcorrect(find(stimRange == StimLevels(level)),:,:,1:7))+(OutOfNum(level)-NumPos(level)).*squeeze(logpincorrect(find(stimRange == StimLevels(level)),:,:,1:7));
    end
    
    [trash I] = PAL_findMax(LLspace);
    paramsInit.alpha = alphas(I(1));
    paramsInit.beta = betas(I(2));
    paramsInit.gamma = .5;
    paramsInit.lambda = lambdas(I(3));

    paramsFree = [1 1 0 1];
    [paramsFit(simulation,4,:) LL(simulation,4) exitflag(simulation,4)] = PAL_PFML_Fit(StimLevels, NumPos, OutOfNum, paramsInit, paramsFree, PF,'lapseLimits',lapseLimits,'searchoptions',options);

    %transform to metric reported by Wichmann & Hill
    paramsFitAdj(simulation,4,1) = PF([paramsFit(simulation,4,1) paramsFit(simulation,4,2) 0 0],.5,'inverse');
    paramsFitAdj(simulation,4,2) = PF([paramsFit(simulation,4,1) paramsFit(simulation,4,2) 0 0],paramsFitAdj(simulation,4,1),'derivative');        
    
    if stepThru
        pause
    end

end

%Display results

%Scatterplots

figure('units','pixels','position',[100 100 1150 700]);

cm = flipud(colormap('jet'));
colormap(cm)

scaling = sqrt(2).^log2(120/Ntrial);

cutoutTlo = log10(8.85) - .36*scaling;
cutoutThi = log10(8.85) + .36*scaling;
cutoutSlo = log10(.118) - 2.1*scaling;
cutoutShi = log10(.118) + 2.1*scaling;

LparamsFitAdj = log10(paramsFitAdj);

for cond = 1:6

    in = LparamsFitAdj(:,cond,1) < cutoutThi & LparamsFitAdj(:,cond,1) > cutoutTlo & LparamsFitAdj(:,cond,2) < cutoutShi & LparamsFitAdj(:,cond,2) > cutoutSlo;
    toPlotT = LparamsFitAdj(in,cond,1);
    toPlotS = LparamsFitAdj(in,cond,2);
    toPlotL = paramsFit(in,cond,4);
    toPlotL(toPlotL > 0.1) = 0.1;

    scatter(toPlotT,toPlotS,3,toPlotL,'filled');
    caxis([0 .1])
    set(gca,'units','pixels','position',[100+mod(cond-1,3)*270 100+(1-floor((cond-1)/3))*270 250 250]);
    axis([cutoutTlo cutoutThi cutoutSlo cutoutShi]);
    yticklabel = [];
    if cond == 1
        text(log10(8.85)-.3*scaling,log10(.118) + 2.4*scaling,'nAPLE','fontsize',14);
        xticklabel = [];
        yticklabel = {'0.025','0.05','0.10','0.2','0.4','0.8'};
        text(log10(8.85)-.52*scaling,log10(.118)-2*scaling,'F^''_{0.5} (''Slope'')','fontsize',14,'horizontalAlignment','center','rotation',90)
    end
    if cond == 2
        text(log10(8.85)-.3*scaling,log10(.118) + 2.4*scaling,'jAPLE','fontsize',14);
        xticklabel = [];
        yticklabel = [];
    end
    if cond == 3  
        text(log10(8.85)-.3*scaling,log10(.118) + 2.4*scaling,'iAPLE','fontsize',14);
        text(log10(8.85)+.4*scaling,log10(.118),'Psi method','horizontalalignment','center','fontsize',16,'rotation',90);
        xticklabel = [];
        yticklabel = [];
        
    end
    if cond == 4       
        xticklabel = {'7','8','9','10','11'};
        yticklabel = {'0.025','0.05','0.10','0.2','0.4','0.8'};
    end
    if cond == 5
        xticklabel = {'7','8','9','10','11'};
        yticklabel = [];
        text(log10(8.85),log10(.118)-3*scaling,'F^{-1}_{0.5} (''Threshold'')','fontsize',14,'horizontalAlignment','center')
    end
    if cond == 6       
        text(log10(8.85)+.4*scaling,log10(.118),'Retested MOCS','horizontalalignment','center','fontsize',16,'rotation',90);
        xticklabel = {'7','8','9','10','11'};
        yticklabel = [];
    end
    set(gca,'xtick',[log10(7) log10(8) log10(9) log10(10) log10(11)],'xticklabel',xticklabel);
    set(gca,'ytick',[log10(.025) log10(.05) log10(.1) log10(.2) log10(.4) log10(.8)],'yticklabel',yticklabel);
    set(gca,'box','on')
    message = ['N = ' int2str(sum(in))];
    text(log10(8.85)-.3*scaling,log10(.118) + 1.8*scaling,message);
    set(gca,'handlevisibility','off');

end

cb = 64:-1:1;
image(cb')
set(gca,'xgrid','off','ygrid','off');
set(gca,'xtick',[],'ytick',[]);
set(gca,'units','pixels','position',[945 100 25 250]);
set(gca,'handlevisibility','off');    
axes;
axis([0 1 0 1]);
axis off
set(gca,'units','pixels','position',[945 100 25 250]);
text(1.4,.5,'Lapse rate','horizontalalignment','center','fontsize',12,'rotation',90);
text(1.3,1,'0.1 (and > 0.1)','horizontalalignment','left','fontsize',12);
text(1.3,0,'0','horizontalalignment','left','fontsize',12);


%Histograms


 
nonpos = paramsFitAdj <= 0 | isnan(paramsFitAdj);%Needed only at low N (if at all)
paramsFitAdj = log10(paramsFitAdj);
paramsFitAdj(nonpos) = -Inf;

for cond = 1:6
    paramsFitAdj(:,cond,3:4) = paramsFit(:,cond,3:4);
    paramsFitAdj(paramsFitAdj(:,cond,1)>=crit(1,2),cond,1) = crit(2,2);
    paramsFitAdj(paramsFitAdj(:,cond,1)<= crit(1,1),cond,1) = crit(2,1);
    paramsFitAdj(paramsFitAdj(:,cond,2)>=crit(3,2),cond,2) = crit(4,2);
    paramsFitAdj(paramsFitAdj(:,cond,2)<= crit(3,1),cond,2) = crit(4,1);
end

for psiMOCS = 0:1

    switch psiMOCS
        case 0
            name = ['Figure 9 pt. I, N = ', int2str(Ntrial), ', Generating lapse rate: ', num2str(lapseGen)];
            titleMessage = 'Psi placement';
        case 1
            name = ['Figure 9 pt. II, N = ', int2str(Ntrial), ', Generating lapse rate: ', num2str(lapseGen)];
            titleMessage = 'Retested as MOCS';
    end

    figure('units','pixels','position',[100 10 750 565],'name',name);
    
%%%%%%%%%%%%%%Lapse Rates

    for nijAPLE = 1:3

        yAdjust = 360-(nijAPLE-1)*180;
        toPlot = squeeze(paramsFitAdj(:,nijAPLE+psiMOCS*3,4));                
        
        switch nijAPLE
            case 1
                headerMessage = '\bfnAPLE';
                lapseBinCenters = .06/80:.06/40:.06-.06/80;
                XL = [0 0.06];
                plotDim = [53 128+yAdjust 289 49];
            case 2
                headerMessage = '\bfjAPLE';
                lapseBinCenters = 0:1/640:.138;
                XL = [0-1/1280 .138];
                plotDim = [53 128+yAdjust 664 49];
            case 3
                headerMessage = '\bfiAPLE';
                lapseBinCenters = 0:1/640:.138;
                XL = [0-1/1280 .138];
                plotDim = [53 128+yAdjust 664 49];
        end
        
        hist(toPlot,lapseBinCenters)
        set(gca,'Units','pixels','position',plotDim,'fontsize',8);
        maxim = max(hist(squeeze(paramsFitAdj(:,nijAPLE+psiMOCS*3,4)),lapseBinCenters));
        axis([XL 0 maxim*1.3]);
        set(gca,'xtick',[0:.01:.13]);
        xlabel('lapse rate estimate','fontsize',8);
        h = findobj(gca,'Type','patch');
        set(h,'FaceColor',[.7 .7 .7],'EdgeColor',[.5 .5 .5])
        line(XL,[0 0],'linewidth',1,'color','k');
        text(.03,maxim*1.5,headerMessage,'horizontalalignment','center','fontsize',12);
        set(gca,'handlevisibility','off');
        
        plot(lapseGen,maxim*1.22,'kv','markerfacecolor','k');
        set(gca,'Units','pixels','position',plotDim);
        axis([XL 0 maxim*1.3]);
        axis off

        if nijAPLE == 1
            text(.08,mean(ylim),titleMessage,'horizontalalignment','left','fontsize',22);
        end
        
        set(gca,'handlevisibility','off');

%%%%%%%%%%%%%%%%%%%Thresholds
        
        binCenters = log10(logspace(crit(2,1),(crit(2,2)),40));

        toPlot = squeeze(paramsFitAdj(:,nijAPLE+psiMOCS*3,1));

        yPos = 49+yAdjust;
        hist(toPlot,binCenters);
        set(gca,'Units','pixels','position',[53 yPos 289 49],'fontsize',8);
        maxim = max(hist(toPlot,binCenters));
        XL = [crit(2,1)-(crit(2,2)-crit(2,1))/20 crit(2,2)+(crit(2,2)-crit(2,1))/20];
        axis([XL 0 maxim*1.3]);
        xtick = get(gca,'xtick');
        xtick(1) = crit(2,1);
        xtick(length(xtick)) = crit(2,2);
        xtick(2:length(xtick)-1) = log10(round(10.^xtick(2:length(xtick)-1)));
        set(gca,'xtick',xtick);
        xticklabel = {};
        for index = 1:length(xtick); xticklabel(index) = cellstr(num2str(10.^xtick(index)));end
        xticklabel(1) = cellstr(['<' num2str(10.^crit(1,1))]);
        xticklabel(length(xticklabel)) = cellstr(['>' num2str(10.^crit(1,2))]);
        set(gca,'xticklabel',xticklabel);
        Median = 10.^median(toPlot);
        message = ['Median = ' num2str(Median,'%5.3f')];
        text(xtick(length(xtick)),maxim,message,'horizontalalignment','right','fontsize',8);
        h = findobj(gca,'Type','patch');
        set(h,'FaceColor',[.7 .7 .7],'EdgeColor',[.5 .5 .5])
        line(XL,[0 0],'linewidth',1,'color','k');
        xlabel('Threshold estimate','fontsize',8);
        set(gca,'handlevisibility','off');

        plot(log10(8.85),maxim*1.22,'kv','markerfacecolor','k');
        set(gca,'Units','pixels','position',[53 yPos 289 49]);
        axis([XL 0 maxim*1.3]);
        axis off
        set(gca,'handlevisibility','off');


%%%%%%%%%%%%%%%%%%%%%%%Slopes


        binCenters = log10(logspace(crit(4,1),crit(4,2),40));

        toPlot = squeeze(paramsFitAdj(:,nijAPLE+psiMOCS*3,2));

        yPos = 49+yAdjust;
        hist(toPlot,binCenters);
        set(gca,'Units','pixels','position',[428 yPos 289 49],'fontsize',8);
        maxim = max(hist(toPlot,binCenters));
        XL = [crit(4,1)-(crit(4,2)-crit(4,1))/20 crit(4,2)+(crit(4,2)-crit(4,1))/20];
        axis([XL 0 maxim*1.3]);
        xtick = get(gca,'xtick');
        xtick = xtick(10.^xtick >= .01);
        xtick(1) = crit(4,1);
        xtick(length(xtick)) = crit(4,2);
        xtick(2:length(xtick)-1) = log10((round(100*(10.^xtick(2:length(xtick)-1))))/100);
        set(gca,'xtick',xtick);
        xticklabel = {};
        for index = 1:length(xtick); xticklabel(index) = cellstr(num2str(10.^xtick(index)));end
        xticklabel(1) = cellstr(['<' num2str(10.^crit(3,1))]);
        xticklabel(length(xticklabel)) = cellstr(['>' num2str(10.^crit(3,2))]);
        set(gca,'xticklabel',xticklabel);
        Median = 10.^median(toPlot);
        message = ['Median = ' num2str(Median,'%5.3f')];
        text(xtick(length(xtick)),maxim,message,'horizontalalignment','right','fontsize',8);
        h = findobj(gca,'Type','patch');
        set(h,'FaceColor',[.7 .7 .7],'EdgeColor',[.5 .5 .5])
        line(XL,[0 0],'linewidth',1,'color','k');
        xlabel('Slope estimate','fontsize',8);
        set(gca,'handlevisibility','off');
        
        plot(log10(.118),maxim*1.22,'kv','markerfacecolor','k');
        set(gca,'Units','pixels','position',[428 yPos 289 49]);
        axis([XL 0 maxim*1.3]);
        axis off
        set(gca,'handlevisibility','off');

    end
    
end


%return warningstates to original settings
warning(warningstates);