%
%PAL_PFHB_inspectFit    Graph data and fit for single subject/condition 
%   combination.
%
%   syntax: PAL_PFHB_inspectFit(pfhb,{optional arguments})
%
%Input:
%   
%   pfhb: Structure created by PAL_PFHB_fitModel
%
%Optional arguments (none are case-sensitive, defaults are indicated by 
%   {}):
%
%   subject: followed by subject ID ({1}, 2, 3, etc.)
%
%   condition: followed by condition ID ({1}, 2, 3, etc.)
%
%   hiDensityCurves: followed by n (some integer in [0 Inf) {0}) adds n 
%       PFs to graph that are randomly sampled from posterior.
%
%   hiDensityRegion: followed by logical {true} (1) or false (0) adds 68%
%       high-density region around PF. More time-consuming than option
%       'hiDensityCurves' but much better-looking.
%
%   hiDensityWidth: followed by scalar changes width of high-density region
%       to the value give. If value given is less than 1, it will be
%       interpreted as a proportion, otherwise as a percentage.
%
%   posteriorPredictiveYs: followed by n (some integer in [0 Inf) {0}) 
%       adds posterior predictive distributions of proportions correct for 
%       each stimulus intensity used. These are derived from n simulated 
%       datasets generated by n PFs that are randomly sampled from the 
%       posterior idstribution.
%
%   centralTendency: followed by additional argument 'mean', {'mode'},
%       'median', uses indicated measure of central tendency.
%
%   all: cycle through all subject x condition combinations. User is
%       prompted for <enter> to move to next or to hit q key followed by
%       enter to quit.
%
%Under certain circumstances, the PF corresponding to the parameter
%   estimates (black line) lies outside of the 68% high-density region
%   (gray band). This is indicative of data for which a high degree of
%   redundancy among parameters exists. See
%   www.palamedestoolbox.org/parameterredundancy.html for information on
%   what is going on and how to apply a band-aid in the present or to avoid 
%   this altogether in the future.
%   
%Introduced: Palamedes version 1.10.0 (NP)
%Modified: Palamedes version 1.10.3, 1.10.5, 1.10.6, 1.10.10, 1.11.12,
% 1.11.13 (See History.m)

function [] = PAL_PFHB_inspectFit(pfhb, varargin)

switch pfhb.model.PF
    case 'logistic'
        PF = @PAL_Logistic;
    case 'cumulativenormal'
        PF = @PAL_CumulativeNormal;
    case 'weibull'
        PF = @PAL_Weibull;
    case 'gumbel'
        PF = @PAL_Gumbel;
    case 'quick'
        PF = @PAL_Quick;
    case 'logquick'
        PF = @PAL_logQuick;
    case 'hyperbolicsecant'
        PF = @PAL_HyperbolicSecant;
end

number = 1;
s = 1;
c = 1;
hiDensityCurves = 0;
hiDensityRegion = true;
hiDensityWidth = 0.68;
posteriorPredictiveYs = 0;
centralTendency = 'mode';
paramNames = {'a','b','g','l'};

if ~isempty(varargin)
    NumOpts = length(varargin);
    n = 1;
    while n <= NumOpts
        valid = 0;
       if strncmpi(varargin{n}, 'subject',4)
            s = varargin{n+1};            
            valid = 1;
            add = 2;
        end                        
        if strncmpi(varargin{n}, 'condition',4)
            c = varargin{n+1};            
            valid = 1;
            add = 2;
        end
        if strncmpi(varargin{n}, 'hiDensityCurves',10)
            hiDensityCurves = varargin{n+1};            
            valid = 1;
            add = 2;
        end
        if strncmpi(varargin{n}, 'hiDensityRegion',10)
            hiDensityRegion = varargin{n+1};            
            valid = 1;
            add = 2;
        end
        if strncmpi(varargin{n}, 'hiDensityWidth',10)
            hiDensityWidth = varargin{n+1};  
            if hiDensityWidth >= 1
                hiDensityWidth = hiDensityWidth/100;
            end
            valid = 1;
            add = 2;
        end        
        if strncmpi(varargin{n}, 'posteriorPredictiveYs',4)
            posteriorPredictiveYs = varargin{n+1};            
            valid = 1;
            add = 2;
        end
        if strncmpi(varargin{n}, 'all',3)
            number = pfhb.model.Ncond*pfhb.model.Nsubj;            
            valid = 1;
            add = 1;
        end        
        if strncmpi(varargin{n}, 'centralTendency',4)
            if ~strcmpi(varargin{n+1},'mean') && ~strcmpi(varargin{n+1},'median') && ~strcmpi(varargin{n+1},'mode')
                warning('PALAMEDES:invalidOption','%s is not a valid value for CentralTendency. Ignored.',varargin{n+1});
            else
                centralTendency = varargin{n+1};
            end
            valid = 1;
            add = 2;
        end
        
        if valid == 0
            warning('PALAMEDES:invalidOption','%s is not a valid option. Ignored.',varargin{n});
            n = n + 1;
        else        
            n = n + add;
        end
    end            
end


for loop = 1:number
    
    if number > 1
        s = floor((loop-1)/pfhb.model.Ncond)+1;
        c = mod((loop-1),pfhb.model.Ncond)+1;
    end

    f = figure('units','normalized','position',[.1 .1 .4 .4],'color','w');
    axes
    hold on
    box on
    xlabel('Stimulus Intensity');
    ylabel('Proportion Positive');

    x = pfhb.data.x(pfhb.data.s == s & pfhb.data.c == c);
    y = pfhb.data.y(pfhb.data.s == s & pfhb.data.c == c);
    n = pfhb.data.n(pfhb.data.s == s & pfhb.data.c == c);
    ppy = [];

    infiniteLim = [x(1) == -Inf x(end) == Inf];
    minx = min(x(~isinf(x)));
    maxx = max(x(~isinf(x)));
    xlimcurve = [minx-(maxx-minx)/5,maxx+(maxx-minx)/5];
    if any(strcmpi(pfhb.model.PF,{'weibull','quick'})) && xlimcurve(1) < 0
        xlimcurve(1) = 0;
    end
    xcurve = linspace(xlimcurve(1),xlimcurve(end),1001);
    numSamples = max(posteriorPredictiveYs, hiDensityCurves);
    
    if numSamples > 0
            sampI = [randi(pfhb.engine.nchains,[1 numSamples]); randi(pfhb.engine.nsamples,[1 numSamples])];

            for sampleI = 1:numSamples

                for paramI = 1:4
                    param = paramNames{paramI};
                    if ~isempty(pfhb.model.(param).cTtoP)
                        samp = pfhb.samples.(param)(sampI(1,sampleI),sampI(2,sampleI),:,s);     
                        paramsSample(sampleI,paramI) = squeeze(samp)'*pfhb.model.(param).cTtoP(:,c);            
                    else
                        paramsSample(sampleI,paramI) = pfhb.model.(param).val;
                    end
                end

            end
            paramsSample(:,2) = 10.^paramsSample(:,2);
            if pfhb.model.gammaEQlambda
                paramsSample(:,3) = paramsSample(:,4);
            end
    
        for hdc = 1:hiDensityCurves
            transp = plot(xcurve, PF(paramsSample(hdc,:),xcurve),'-','linewidth',2,'color',[.5 .5 .5]);
            if strcmp(pfhb.machine.environment,'matlab')
                transp.Color(4) = 1/(log(hiDensityCurves)+1);
            end
        end
    end

    if hiDensityRegion
        xloc = xcurve([0:100]*(1000/100)+1);
        paramsInGrid = cellstr({});
        samples = [];
        for param = paramNames
            param = char(param);
            if isfield(pfhb.samples,param) | isfield(pfhb.samples,strcat(param,'_actual'))
                if isfield(pfhb.samples,strcat(param,'_actual'))
                    param = strcat(param,'_actual');
                end
                samples = [samples, reshape(pfhb.samples.(param)(:,:,c,s),[pfhb.engine.nsamples.*pfhb.engine.nchains,1])];
                paramsInGrid{length(paramsInGrid)+1} = param(1);
            end
        end                  

        gridsize = 2^(24/length(paramsInGrid)); %contain combinatorial explosion

        for paramI = 1:length(paramsInGrid)
            [grid, pdf] = PAL_kde(samples(:,paramI));
            trimmedRange = PAL_hdi(grid,pdf,99);                        %avoid grid waste
            trimmedRange = [trimmedRange(1,1), trimmedRange(end,2)];
            samples = samples(samples(:,paramI)>trimmedRange(1) & samples(:,paramI)<trimmedRange(2),:);
        end       
            
        leftAfterTrim = length(samples)/(pfhb.engine.nchains*pfhb.engine.nsamples);

        [gridPosterior edges] = PAL_ndhisto(samples,gridsize);

        kernelGridAxes = {};
        for dim = 1:length(paramsInGrid)
            kernelGridAxes{dim} = -gridsize/16:gridsize/16;
        end
        
        kernelGrids = cell(1, length(paramsInGrid));            
        [kernelGrids{:}] = ndgrid(kernelGridAxes{:});

        kernel = ones(size(kernelGrids{1}));
        for dim = 1:length(paramsInGrid)
            sd = std(samples(:,dim))/(5*(edges(2,dim)-edges(1,dim)));
            kernel = kernel.*PAL_pdfNormal(kernelGrids{dim},0,sd);
        end
        gridPosterior = convn(gridPosterior,kernel,'same');
        gridPosterior = gridPosterior(:)/sum(gridPosterior(:));

        [gridPosterior, I] = sort(gridPosterior,'descend');
        gridPosterior = cumsum(gridPosterior);

        hdicutoff = hiDensityWidth/leftAfterTrim;
        paramI = 0;
        paramNamesFull = {'alpha','beta','gamma','lambda'};
        for paramIndexNo = 1:4
            paramIndex = paramNames{paramIndexNo};
            where = find(strcmp(paramIndex,paramsInGrid));
            if where
                edgeIs = mod((ceil(I(gridPosterior<=hdicutoff)./(gridsize^paramI))-1),gridsize) + 1;
                paramI = paramI+1;                    
                paramsInHDR.(paramNamesFull{paramIndexNo}) = edges(edgeIs,paramI)+(edges(2,paramI)-edges(1,paramI))/2;
            else
                paramsInHDR.(paramNamesFull{paramIndexNo}) = pfhb.model.(paramNames{paramIndexNo}).val.*ones(length(I(gridPosterior<=hdicutoff)),1);
            end
        end
        paramsInHDR.beta = 10.^paramsInHDR.beta;
        if pfhb.model.gammaEQlambda
            paramsInHDR.gamma = paramsInHDR.lambda;
        end
        y_s = PF(paramsInHDR,repmat(xloc,[length(paramsInHDR.beta),1]));
        patch([xloc fliplr(xloc)], [min(y_s) fliplr(max(y_s))], [.7 .7 .7],'edgecolor',[.7 .7 .7],'facealpha',.5);        
    end
    
    for paramI = 1:4
        param = paramNames{paramI};
        if ~isempty(pfhb.model.(param).cTtoP)
            params(paramI) = pfhb.summStats.(param).(centralTendency)(:,s)'*pfhb.model.(param).cTtoP(:,c);
        else
            params(paramI) = pfhb.model.(param).val;
        end
    end
    params(2) = 10.^params(2);
    if pfhb.model.gammaEQlambda
        params(3) = params(4);
    end

    plot(xcurve, PF(params,xcurve),'k-','linewidth',2);

    for ix = 1:length(x)
        if ~isinf(x(ix))            
            S = scatter(x(ix),y(ix)./n(ix),500*sqrt(n(ix)./max(n)),[0.8 0 0.2],'o','filled');
            S.MarkerFaceAlpha = 0.5;
        end
    end

    if any(infiniteLim)
        xlim = get(gca,'xlim');
        xt = get(gca,'xtick');
        xticklabel = get(gca,'xticklabel');
        xtick = xt(xt > xlimcurve(1) & xt < xlimcurve(2));
        xticklabel = xticklabel(xt > xlimcurve(1) & xt < xlimcurve(2));
        if infiniteLim(1)
            xlim(1) = xlimcurve(1) - (xlimcurve(2)-xlimcurve(1))/4;
            x(1) = .8*xlim(1) + .2*xlimcurve(1);
            xtick = [x(1) xtick];
            xticklabel = [{-Inf'}; xticklabel];
            for hdc = 1:hiDensityCurves

                if ~isempty(pfhb.model.g.cTtoP)
                    samp = pfhb.samples.g(sampI(1,hdc),sampI(2,hdc),:,s);     
                    value = squeeze(samp)'*pfhb.model.g.cTtoP(:,c);            
                else
                    value = pfhb.model.g.val;
                end
                transp = line([x(1) .5*xlim(1) + .5*xlimcurve(1)],[value value], 'linewidth',2,'color',[.5 .5 .5]);
                if strcmp(pfhb.machine.environment,'matlab')
                    transp.Color(4) = 1/(log(hiDensityCurves)+1);
                end
            end
            line([x(1) .5*xlim(1) + .5*xlimcurve(1)],[params(3) params(3)], 'linewidth',2,'color',[0 0 0]);
            S = scatter(x(1),y(1)./n(1),500*sqrt(n(1)./max(n)),[0.8 0 0.2],'o','filled');
            S.MarkerFaceAlpha = 0.5;

        end
        if infiniteLim(2)
            xlim(2) = xlimcurve(2) + (xlimcurve(2)-xlimcurve(1))/4;
            x(end) = .8*xlim(2) + .2*xlimcurve(2);
            xtick = [xtick x(end)];
            xticklabel = [xticklabel; {Inf'}];
            for hdc = 1:hiDensityCurves

                if ~isempty(pfhb.model.l.cTtoP)
                    samp = pfhb.samples.l(sampI(1,hdc),sampI(2,hdc),:,s);     
                    value = squeeze(samp)'*pfhb.model.l.cTtoP(:,c);            
                else
                    value = pfhb.model.l.val;
                end
                transp = line([x(end) .5*xlim(2) + .5*xlimcurve(2)],[1-value 1-value], 'linewidth',2,'color',[.5 .5 .5]);
                if strcmp(pfhb.machine.environment,'matlab')
                    transp.Color(4) = 1/(log(hiDensityCurves)+1);
                end
            end        
            line([x(end) .5*xlim(2) + .5*xlimcurve(2)],[1-params(4) 1-params(4)], 'linewidth',2,'color',[0 0 0]);
            S = scatter(x(end),y(end)./n(end),500*sqrt(n(end)./max(n)),[0.8 0 0.2],'o','filled');
            S.MarkerFaceAlpha = 0.5;
        end

    else
        xlim = xlimcurve;
        xtick = get(gca,'xtick');
        xticklabel = get(gca,'xticklabel');
    end
   
    if posteriorPredictiveYs > 0
        for ppyl = 1:posteriorPredictiveYs
            ppy(ppyl,:) = PAL_PF_SimulateObserverParametric(paramsSample(ppyl,:), x, n, PF);
        end
        for ix = 1:length(n)
            stepsize = (ceil(n(ix)./100)).*1./n(ix);
            [ppy_n ppy_y] = hist(ppy(:,ix)./n(ix),[0:stepsize:1]);
            for ppy_i = 1:length(ppy_n)
                if ppy_n(ppy_i) > 0
                    plot(x(ix), ppy_y(ppy_i),'ko','markersize',30*sqrt((ppy_n(ppy_i)./max(ppy_n)).*(n(ix)./max(n))),'linewidth',1); %posteriorPredictiveYs
                end
            end
        end
    end
    
    set(gca, 'xlim', xlim,'ylim',[0 1]);
    set(gca,'xtick',xtick,'xticklabel',xticklabel)

    text(xlim(1) + (xlim(2)-xlim(1))./10, 1.05,['Subject: ',int2str(s),'   Condition: ', int2str(c)]);
    text(xlim(1) + (xlim(2)-xlim(1))*.65, .3,['Parameter values (',centralTendency,'):']);
    text(xlim(1) + (xlim(2)-xlim(1))*.65, .25,['\alpha: ',num2str(params(1))]);
    text(xlim(1) + (xlim(2)-xlim(1))*.65, .2,['\beta: ',num2str(params(2))]);
    text(xlim(1) + (xlim(2)-xlim(1))*.65, .15,['\gamma: ',num2str(params(3))]);
    text(xlim(1) + (xlim(2)-xlim(1))*.65, .1,['\lambda: ',num2str(params(4))]);
    
    if number > 1
        q = input('Hit <enter> to see next or type ''q'' (without quotes) followed by <enter> to quit\n','s');
        close(f); 
        if q == 'q'
            break;
        end
    end

end