%
%PAL_PFHB_MultipleSubjectsSingleCondition_Demo  Demonstrates use of 
%PAL_PFHB_fitModel.m to fit multiple psychometric functions to data derived 
%from multiple subjects testing in a single condition using a Bayesian 
%Hierarchical model. Data are collected using the psi-marginal method (this 
%takes a little bit of time). The model that is fitted is shown here:
%www.palamedestoolbox.org/hierarchicalbayesian.html)
%JAGS (http://mcmc-jags.sourceforge.net/) or cmdSTAN ('command Stan')
%(https://mc-stan.org/users/interfaces/cmdstan.html) must first be
%installed before this will work. JAGS or Stan will perform the MCMC
%sampling of the posterior.
%Note that in order for MCMC sampling to converge you'll need at least one
%of these conditions to exist:
%1. High number of trials
%2. Informative priors (default priors are not informative)
%3. High number of participants
%4. Low number of free parameters
%5. Luck
%
%NP (May 2019)

clear all;
    
engine = input('Use Stan or JAGS (either must be installed from third party first, see PAL_PFHB_fitModel for information)? Type stan or jags: ','s');

disp(['Generating some data using adaptive method....',char(10)]);

%Setting up simulations. Note the inclusion of a stimulus intensity at an asymptotic level of performance.
%This is a good idea when freeing lapse rate (see www.palamedestoolbox.org/parameterredundancy.html)

n_subjects = 12;
n_trials_per_subject = 480;
intensities = [linspace(5,15,11), 25];
generating_locations = 10 + randn(1,n_subjects);    %Location parameters ('thresholds') in population are normally distributed with mean = 10 and sd = 1
generating_logslopes = .2 * randn(1,n_subjects);    %Log(slope) parameters in population are normally distributed with mean = 0 and sd = .2
if exist('betarnd','file')                         %requires Matlab's statistics and machine-learning toolbox
    generating_lapse_rates = betarnd(3,97,1,n_subjects);    %Lapse rate parameters in population are beta distributed with a = 3 and b = 97
                                                   %(i.e., with mean a/(a+b) = 0.03 and concentration a + b = 100)
else
    generating_lapse_rates = [0.0205, 0.0116, 0.0350, 0.0301, 0.0171, 0.0453, 0.0600, 0.0395, 0.0412, 0.0654, 0.0181, 0.0062];
end

%Generate some data using adaptive psi-marginal method (type help PAL_AMPM_setupPM
%for more information)
grain = 51;

%Define parameter ranges to be included in psi-marginal method's posterior
priorAlphaRange = linspace(5, 15,grain);
priorBetaRange =  linspace(-1,1,grain); %Use log10 transformed values of beta (slope) parameter
priorGammaRange = 0.5; 
priorLambdaRange = 0:.01:.1; 

%Simulate 2-AFC experiment using Psi-marginal method to control stimulus
%placement. We marginalize the lapse rate so that the adaptive methods only
%targets lapse rate when this is the best way to increase information about
%location and slope parameters.
%Set up prior for adaptive method:

%Create prior for psi-marginal method
[a b g l] = ndgrid(priorAlphaRange,priorBetaRange,priorGammaRange,priorLambdaRange);
prior = PAL_pdfNormal(a,10,2).*PAL_pdfNormal(b,0,.4).*PAL_pdfBeta(l,1,10);    %prior used for psi-marginal method
prior = prior./sum(sum(sum(sum(prior))));

data.x = [];
data.s = [];
data.y = [];
data.n = [];

for s = 1:n_subjects

    %Initialize PM structure
    PM = PAL_AMPM_setupPM('priorAlphaRange',priorAlphaRange,...
                          'priorBetaRange',priorBetaRange,...
                          'priorGammaRange',priorGammaRange,...
                          'priorLambdaRange',priorLambdaRange,...
                          'numtrials',n_trials_per_subject,...
                          'PF' , @PAL_Logistic,...
                          'stimRange',intensities,...
                          'marginalize',['lapse'],...   %treat lapse rate as nuisance parameter (i.e., method will not optimize estimation of lapse rate for it's own sake
                              'prior',prior);           %but will target lapse rate if this is most efficient way to gain information about primary parameters[location, slope])
                                                        %see https://doi.org/10.1167/13.7.3


    %trial loop
    while PM.stop ~= 1

        response = rand(1) < PAL_Logistic([generating_locations(s), 10.^generating_logslopes(s), 0.5, generating_lapse_rates(s)], PM.xCurrent);    %simulate response

        %update PM based on response
        PM = PAL_AMPM_updatePM(PM,response);

    end

    [xG yG nG] = PAL_PFML_GroupTrialsbyX(PM.x(1:end-1),PM.response,ones(size(PM.response)));

    data.x = [data.x xG];
    data.y = [data.y yG];
    data.n = [data.n nG];
    data.s = [data.s s*ones(size(xG))];

end

%Analyze data
disp(['Analyzing data....',char(10)]);

%Perform fit
[pfhb] = PAL_PFHB_fitModel(data,'engine',engine,'nsamples',15000);

%Inspect fit for one subject (by default, this will be subject '1', use optional arguments to inspect others)
PAL_PFHB_inspectFit(pfhb);

%Inspect posterior for location mean hyperparameter ('amu', compare to generating value of 10)
PAL_PFHB_inspectParam(pfhb,'amu');

%Inspect posterior for location sd hyperparameter ('asigma', compare to generating value of 1)
PAL_PFHB_inspectParam(pfhb,'asigma');

%Inspect posterior for log(slope) mean hyperparameter ('bmu', compare to generating value of 0)
PAL_PFHB_inspectParam(pfhb,'bmu');

%Inspect posterior for log(slope) sd hyperparameter ('bsigma', compare to generating value of .2)
PAL_PFHB_inspectParam(pfhb,'bsigma');

%Inspect posterior for lapse rate mean hyperparameter ('lmu', compare to generating value of 0.03)
PAL_PFHB_inspectParam(pfhb,'lmu');

%Inspect posterior for lapse rate concentration hyperparameter ('lkappa', compare to generating value of 100)
PAL_PFHB_inspectParam(pfhb,'lkappa');

%Inspect violin plots. The default for this analysis is to display violins for the mean hyperparameters ('amu', 'bmu', 'lmu')
%Use optional arguments to display violins for other parameters.
PAL_PFHB_drawViolins(pfhb);
