%
%PAL_PFHB_MultipleSubjectsandConditions_Demo  Demonstrates use of 
%PAL_PFHB_fitModel.m to fit multiple psychometric functions to data derived 
%from multiple subjects testing in multiple conditions using a Bayesian 
%criterion. Data are collected using the psi-marginal method (this takes a 
%little bit of time). The model that is fitted is shown here:
%www.palamedestoolbox.org/hierarchicalbayesian.html)
%JAGS (http://mcmc-jags.sourceforge.net/) or cmdSTAN ('command Stan')
%(https://mc-stan.org/users/interfaces/cmdstan.html) must first be
%installed before this will work. JAGS or Stan will perform the MCMC
%sampling of the posterior. Some of the optional tweaks are demonstrated
%here.
%Note that in order for MCMC sampling to converge you'll need at least one
%of these conditions to exist:
%1. High number of trials
%2. Informative priors (default priors are not informative)
%3. High number of participants
%4. Low number of free parameters
%5. Luck
%
%NP (May 2019)

clear all;

engine = input('Use Stan or JAGS (either must be installed from third party first, see PAL_PFHB_fitModel for information)? Type stan or jags: ','s');

disp(['Generating some data using adaptive method. This may take a while....',char(10)]);

n_conditions = 4;
n_subjects = 6;
n_trials_per_subject = 480;

%Generate some generating parameter values to use in simulation:
location_condition_effect = repmat(randn(n_conditions,1),[1,n_subjects]);
location_subject_effect = repmat(randn(1,n_subjects),[n_conditions,1]);
generating_locations = location_condition_effect+location_subject_effect;

logslope_condition_effect = repmat(.1*randn(n_conditions,1),[1,n_subjects]);
logslope_subject_effect = repmat(.2*randn(1,n_subjects),[n_conditions,1]);
generating_logslopes = logslope_condition_effect+logslope_subject_effect;

if exist('betarnd','file')                                  %requires Matlab's statistics and machine-learning toolbox
    generating_lapse_rates = betarnd(3,97,1,n_subjects);    %Lapse rate parameters in simulated population are beta distributed with a = 3 and b = 97
                                                            %(i.e., with mean a/(a+b) = 0.03 and concentration a + b = 100)
                                                            %Generating lapse rates are equal across conditions for each subject
else
    generating_lapse_rates = [0.0205, 0.0116, 0.0350, 0.0301, 0.0171, 0.0453];
end

%Generate some data using adaptive psi-method (type help PAL_AMPM_setupPM
%for more information)
grain = 51;

%Define parameter ranges to be included in posterior of psi-marginal method
priorAlphaRange = linspace(-2,2,grain); %Location ('threshold') parameter
priorBetaRange =  linspace(-1,1,grain); %Use log10 transformed values of beta (slope) parameter in PF
priorGammaRange = 0.25;                 %Lower asymptote ('guess rate')4-AFC task
priorLambdaRange = 0:.01:.1;            %1 - upper asymptote ('lapse rate')

%Stimulus intensities available to psi-marginal method
intensities = [linspace(-4,4,11), 10];

[a b g l] = ndgrid(priorAlphaRange,priorBetaRange,priorGammaRange,priorLambdaRange);

prior = PAL_pdfNormal(a,0,1).*PAL_pdfNormal(b,0,1).*PAL_pdfBeta(l,1,10);    %prior for psi-marginal method
prior = prior./sum(sum(sum(sum(prior))));

data.x = [];
data.s = [];
data.c = [];
data.y = [];
data.n = [];

for c = 1:n_conditions
    
    for s = 1:n_subjects
        
        generating_parameters = [generating_locations(c,s), 10.^generating_logslopes(c,s), 0.25, generating_lapse_rates(s)];   %parameter values [alpha, beta, gamma, lambda] (or [threshold, slope, guess, lapse]) used to simulate observer

        %Initialize PM structure
        PM = PAL_AMPM_setupPM('priorAlphaRange',priorAlphaRange,...
                              'priorBetaRange',priorBetaRange,...
                              'priorGammaRange',priorGammaRange,...
                              'priorLambdaRange',priorLambdaRange,...
                              'numtrials',n_trials_per_subject,...
                              'PF' , @PAL_Logistic,...
                              'stimRange',intensities,...
                              'marginalize',['lapse'],...   %treat lapse rate as nuisance parameter (i.e., method will not optimize estimation of lapse rate for it's own sake
                              'prior',prior);               %but will target lapse rate if this is most efficient way to gain information about primary parameters[location, slope])
                                                            %see https://doi.org/10.1167/13.7.3
                       
        %trial loop
        while PM.stop ~= 1

            response = rand(1) < PAL_Logistic(generating_parameters, PM.xCurrent);    %simulate observer

            %update PM based on response
            PM = PAL_AMPM_updatePM(PM,response);

        end
        [xG yG nG] = PAL_PFML_GroupTrialsbyX(PM.x(1:end-1),PM.response,ones(size(PM.response)));

        data.x = [data.x xG];
        data.y = [data.y yG];
        data.n = [data.n nG];
        data.c = [data.c c*ones(size(xG))];
        data.s = [data.s s*ones(size(xG))];

    end
end

disp(['Analyzing data....',char(10)]);

%Run analysis
%-fix lower asymptote ('guess rate') at 0.25 (4-AFC task)
%-use 15000 samples from posterior (instead of default 5000)
[pfhb] = PAL_PFHB_fitModel(data,'engine',engine,'guess','fixed',0.25,'nsamples',15000);

%The following will, by default, display violin plots for the
%hyperparameters (when multiple subjects are included in analysis)
PAL_PFHB_drawViolins(pfhb);

%The following shows violin plots for subject-level location parameters
%across (arbitrarily) subjects 1, 3, 4, and 5 and conditions 2, 3, and 4
PAL_PFHB_drawViolins(pfhb,'a','subjects',[1,3:5],'conditions',[2:4])

%Inspect posterior and diagnostics for hyper-parameter corresponding to the mean location
%(aka 'threshold') parameter across subjects for (arbitrarily) condition 2
PAL_PFHB_inspectParam(pfhb,'amu','condition',2);

%Inspect posterior for the difference between the location parameter for
%subject 1 in condition 1 and this subject's lapse rate (not particularly 
%meaningful) but also a scatterplot between these two parameters which can 
%be useful to detect redundancy among parameters 
%(see www.palamedestoolbox.org/parameterredundancy.html)
PAL_PFHB_inspectParam(pfhb,'a','subject',1,'condition',1,'l','subject',1,'effect',1);

%Inspect data with fit for arbitrarily subject 3, condition 2, using the mean of the
%posteriors as estimates for parameters
PAL_PFHB_inspectFit(pfhb,'subject',3,'condition',2,'centraltendency','mean');
