%
%PAL_PFHB_SinglePF_Demo  Demonstrates use of PAL_PFHB_fitModel.m to fit a
%single psychometric function to some data using a Bayesian criterion. The 
%model that is fitted is shown here:
%www.palamedestoolbox.org/hierarchicalbayesian.html)
%JAGS (http://mcmc-jags.sourceforge.net/) or cmdSTAN ('command Stan')
%(https://mc-stan.org/users/interfaces/cmdstan.html) must first be
%installed before this will work. JAGS or Stan will perform the MCMC
%sampling of the posterior.
%Note that in order for MCMC sampling to converge you'll need at least one
%of these conditions to exist:
%1. High number of trials (this is why this fit will likely converge)
%2. Informative priors (default priors are hardly informative)
%3. High number of participants
%4. Low number of free parameters
%5. Luck
%
%NP (May 2019)

clear all;

engine = input('Use Stan or JAGS (either must be installed from third party first, see PAL_PFHB_fitModel for information)? Type stan or jags: ','s');

data.x = [-4:4,10];
data.n = [40*ones(size(data.x))];
data.y = PAL_PF_SimulateObserverParametric([0 1 .5 .03],data.x,data.n,@PAL_Logistic);


% By default, palamedes.pfhb_fit_model: -assumes logistic form of PF
%                                       -fixes guess rate at 0.5 (as would be appropriate for 2AFC task)
%                                       -considers location (aka 'threshold'), slope and lapse rate to be free parameters
%                                       -priors on location and log(slope) parameters as described here: www.palamedestoolbox.org/guidedpriors.html
%                                       -beta prior on lapse rate with mean = 1/11 and concentration 11 (equivalent to a = 1, b = 10)


%Use defaults (except for engine):
pfhb = PAL_PFHB_fitModel(data,'engine',engine,'nsamples',50000);   %accepts additional optional arguments (type 'help PAL_PFHB_fitModel')


%pfhb is now a structure with all the information about the model (pfhb['model']), summary statistics of fit (pfhb['summStats']) and much more

%Inspect summary statistics for location parameter ('a'):
disp(['Summary statistics for location (aka ''threshold'') parameter:',char(10)])
disp(pfhb.summStats.a)

%Display data with fitted PF:
PAL_PFHB_inspectFit(pfhb);       %accepts optional arguments (type 'help PAL_PFHB_inspectFit')

%Display the posterior distribution for a specific parameter, with 68% credible (or high-density) interval, and some diagnostics:
PAL_PFHB_inspectParam(pfhb);     %accepts additional optional arguments (type 'help PAL_PFHB_inspectParam')

%Create figure showing violin plots:
PAL_PFHB_drawViolins(pfhb);      %accepts optional arguments (type 'help PAL_PFHB_drawViolins')