%
%PAL_PFHB_SingleSubjectMultipleConditions_Demo  Demonstrates use of 
%PAL_PFHB_fitModel.m to fit multiple psychometric functions to data derived 
%from a single subject testing in multiple conditions using a Bayesian 
%criterion. The model that is fitted is shown here:
%www.palamedestoolbox.org/hierarchicalbayesian.html)
%JAGS (http://mcmc-jags.sourceforge.net/) or cmdSTAN ('command Stan')
%(https://mc-stan.org/users/interfaces/cmdstan.html) must first be
%installed before this will work. JAGS or Stan will perform the MCMC
%sampling of the posterior. Some of the optional tweaks are demonstrated
%here.
%Three analyses are performed. The first estimates the 'raw' location 
%('threshold') and slope parameters. The second reparemeterizes the 
%location ('threshold') parameters into three orthogonal 'effects' in order 
%to investigate differences between conditions. The third reduces model 
%complexity by constraining slope parameters to be equal between the three 
%conditions.
%Note that in order for MCMC sampling to converge you'll need at least one
%of these conditions to exist:
%1. High number of trials
%2. Informative priors (default priors are not informative)
%3. High number of participants
%4. Low number of free parameters
%5. Luck
%
%NP (May 2019)

clear all;

engine = input('Use Stan or JAGS (either must be installed from third party first, see PAL_PFHB_fitModel for information)? Type stan or jags: ','s');

x = [-4:1:4];
n = 100*ones(size(x));

data.x = [];
data.y = [];
data.n = [];
data.c = [];

%Set generating parameters to simulate a three-condition experiment 
%(location, slope, lower asymptote ('guess rate'), upper
%asymptote ('lapse rate'). These parameter values are consistent with an
%appearance-based task for which both the lower and upper asymptote are
%determined by 'lapses'.
generating_params = [-1 1 .02 .02;      
                      1 1 .02 .02;
                      1 1 .02 .02];

%Simulate some data
for condition = 1:3
    data.x = [data.x x];
    data.y = [data.y PAL_PF_SimulateObserverParametric(generating_params(condition,:),x,n,@PAL_Logistic)];
    data.n = [data.n n];
    data.c = [data.c condition*ones(size(n))];
end

%Basic fit. Setting 'gammaEQlambda' to logical true constrains the gamma
%and lambda parameters to have equal value.
pfhb = PAL_PFHB_fitModel(data,'gammaEQlambda',true,'engine',engine,'nsamples',10000);
%Inspect data and fit in one condition
PAL_PFHB_inspectFit(pfhb,'condition',2);
%The following generates warning and explanation. Included here to stress
%that constraining the lapse rate across conditions is the default.
%Warning also makes explicit that what is plotted is the 'reparameter' that
%is a linear combination of the lapse rates in the three conditions that
%corresponds to the average lapse rate in the three conditions
%(i.e., 0.33333*lapse rate 1 + 0.33333*lapse rate 2 + 0.33333*lapse rate 3
msg = ['The code that generates the following warning is intentionally included to draw attention to the fact that',char(10),'constraining the lapse rate across conditions is the default:'];
disp(msg);
PAL_PFHB_inspectParam(pfhb,'l','condition',2);

PAL_PFHB_drawViolins(pfhb);
%Note that in the violin plot the x-axis for the lapse rate parameter is
%labeled 'effect', not 'condition' to reflect the idea that not every
%condition gets its own unique lapse rate estimate.

%The following shows diagnostics and derives summary statistics for the 
%difference between the location parameters (aka 'threshold') in conditions 
%1 and 2. Also shows scatter plot between these two parameters.
PAL_PFHB_inspectParam(pfhb,'a','condition',1,'a','condition',2);
%The following shows diagnostics and derives summary statistics for the 
%difference between the location parameter (aka 'threshold') in condition 
%1 and the lapse rate(which are not interesting for these two parameters). 
%Also shows scatter plot between these two parameters (which is interesting,
%see www.palamedestoolbox.org/parameterredundancy.html)
PAL_PFHB_inspectParam(pfhb,'a','condition',1,'l');

%In the next analysis, the location parameters are reparameterized using 
%the Model Matrix M below (see www.palamedestoolbox.org/modelmatrix.html). 
%E.g., The first 'reparameter' corresponds to the mean of the location 
%parameters across conditions. The second 'reparameter' would compare
%location parameter in condition 1 versus the average of the location 
%parameters in condition 2 and 3. The third would compare location 
%parameters between conditions 2 and 3. 
M = [1/3 1/3 1/3; -1 1/2 1/2; 0 -1 1];
pfhb = PAL_PFHB_fitModel(data,'gammaEQlambda',true,'engine',engine,'a',M,'nsamples',10000);
PAL_PFHB_inspectFit(pfhb,'condition',2);
%Inspect posterior for e.g., the third 'reparameter' on location. Remember
%that it corresponds to difference between condition 2 and 3. Generating
%values were identical in conditions 2 and 3 so true value for 
%'reparameter' equals 0 and posterior should have high density at value of 
%0.
PAL_PFHB_inspectParam(pfhb, 'a', 'effect', 3);
%Note that the axis for the location parameter in the violin plot is now 
%labeled 'effect', not 'condition'.
PAL_PFHB_drawViolins(pfhb);
%To inspect the 'raw' location parameters in each condition, use 'a_actual'
%in PAL_PFHB_drawViolins or PAL_PFHB_inspectParam, e.g.:
PAL_PFHB_drawViolins(pfhb,'a_actual');

%In the next analysis, slopes are constrained to be equal across
%conditions. Equivalent to using model matrix M = [1 1 1]/3 for 'b'
pfhb = PAL_PFHB_fitModel(data,'gammaEQlambda',true,'engine',engine,'b','constrained','nsamples',10000);
%print the single, shared log(slope) value for all conditions:
disp(['The shared value for log(slope) used in all three conditions is: ', num2str(pfhb.summStats.b.mean)]);
%Note that in the violin plots for parameters, the horizontal axis for
%slope is now also labeled 'effect', not 'condition'
PAL_PFHB_drawViolins(pfhb);

