%PAL_PFML_FailedMLFits_Demo demonstrates how maximum-likelihood fits may 
%fail due to reasons described here:
%www.palamedestoolbox/understandingfitting.html
%and how Palamedes responds when this is the case.
%
%NP (November 2021)

clear all;

msg =['Maximum-Likelihood fits fail if the likelihood function has no global maximum located at finite parameter values.',char(10)];
msg =[msg 'In such cases, there is either a step function or a constant function that can be approached by a sigmoid, but',char(10)];
msg =[msg 'never exactly matched. Palamedes can detect these situations and tell you when this is the case and whether a step',char(10)];
msg =[msg 'function or constant function fits better than a sigmoid. Note that it is not the fitting that fails, but rather that',char(10)];
msg =[msg 'the data are deficient: no maximum exists in the likelihood. There are three distinct types of these failed fits.',char(10)];
msg =[msg 'Palamedes calls these ''scenarios''. The scenarios are shown here: www.palamedestoolbox.org/deficientdatascenarios.html',char(10)];
msg =[msg 'Think you can spot the deficient data by eye? Give it a shot here. First you will see some synthetic data (generated by',char(10)];
msg =[msg 'a logistic with location (''threshold'') equal to 0, slope equal to 1, guess rate equal to 0.5 and lapse rate equal to',char(10)];
msg =[msg '0.02). The generating function is shown with the data in black. Each data point represents 10 trials.',char(10)];
msg =[msg 'Take a stab at it, then hit the space bar to reveal the best fit and a graphic of the likelihood function (but note that',char(10)];
msg =[msg 'when the lapse rate is free, the likelihood function shown is collapsed across lapse rates). Hit space bar again for a',char(10)];
msg =[msg 'new dataset. This demo is on a never-ending loop. Use control-c to quit. There are two options: Fit with a free lapse',char(10)];
msg =[msg 'rate constrained to [0 0.06] or use a fixed lapse rate (at 0.02).',char(10)];
disp(msg);
lapseFree = input('Use a fixed (0) or free (1) lapse rate? (0/1): ');

f1 = figure('units','pixels','position',[500 100 400 600],'color','w');

%Synthetic observer:
PF = @PAL_Logistic;
paramsGen = [0 1 .5 .02];

%Which parameters are free during fit ([location('threshold') slope guess
%lapse])
paramsFree = [1 1 0 lapseFree];

%Stimulus intensities (note that stimulus placement is very reasonable
%relative to generating function)
xG = [-2:1:2];
%Num trials for each intensity (note that number of trials is not
%atypical of practice in research)
oG = 10*ones(size(xG));

%For plotting:
xFine = linspace(PF([0 1],.01,'inverse'),PF([0 1],.99,'inverse'),1001);

%Brute force search grid for starting point of Nelder-Mead
grid.alpha = linspace(min(xFine),max(xFine),201);
grid.beta = 10.^linspace(-2,2,201);
grid.gamma = .5;
if paramsFree(4)
    grid.lambda = linspace(0,.06,51);
else
    grid.lambda = 0.02;
end

%Parameter space
LLspace = zeros(length(grid.alpha),length(grid.beta),length(grid.gamma),length(grid.lambda));

while 1 %forever
    
    %Simulate responses
    pG = PAL_PF_SimulateObserverParametric(paramsGen,xG,oG,PF);
    
    
    if lapseFree
        [params LL scenario] = PAL_PFML_Fit(xG, pG, oG, grid, paramsFree, PF,'lapselimits',[0 .06]);
    else %to avoid warning being thrown
        [params LL scenario] = PAL_PFML_Fit(xG, pG, oG, grid, paramsFree, PF);
    end
    
    %Plot data and generating function
    ax1 = axes('units','normalized','position',[.15 .67 .8 .3]);
    hold on
    colors = get(ax1,'colororder');
    xlim = [xFine(1) xFine(end)];
    line([xlim(1) xlim(2)], [grid.gamma grid.gamma],'linestyle',':','linewidth',2,'color','k')
    text(xlim(2)-(xlim(2)-xlim(1))/10,.55,'\gamma','horizontalalignment','right');

    plot(ax1,xG,pG./oG,'o','markersize',9,'color',colors(2,:),'markerfacecolor',colors(2,:));
    plot(ax1,xFine,PF(paramsGen,xFine),'k-');
    set(ax1,'ylim',[0 1],'ytick',0:.5:1,'xlim',xlim,'xtick',[-4:2:4],'fontsize',11,'box','on')
    ylabel(ax1,'prop correct');

    pause

    %Plot best-fitting function and likelihood function
    if scenario > -3
        plot(ax1,xFine,PF(params,xFine),'-','color',colors(1,:),'linewidth',2)
    else
        plot(ax1,xFine,ones(size(xFine)).*sum(pG)./sum(oG),'-','color',colors(1,:),'linewidth',2)
    end

    if scenario == -1
        plot(ax1,params(1),pG(xG==params(1))./oG(xG==params(1)),'o','markersize',15,'color',colors(1,:),'markerfacecolor','w','linewidth',2);
    end
    plot(ax1,xG,pG./oG,'o','markersize',9,'color',colors(2,:),'markerfacecolor',colors(2,:));
    text(0,.1,['Scenario: ',int2str(scenario)]);

    hold off

    ax2 = axes('units','normalized','position',[.15 .09 .8 2*.8/3]);
    set(ax2,'ylim',[-2 2],'ytick',-2:1:2,'yticklabel',{'.01','.1','1','10','100'},'xlim',[xFine(1) xFine(end)],'xtick',[-4:2:4],'fontsize',11)
    xlabel('location');
    ylabel('slope')

    [params LLbf LLspace] = PAL_PFML_BruteForceFit(xG,pG,oG,grid,PF);
    LLspace = sum(10.^LLspace,4);
    LLspace = permute(LLspace,[2 1]);
    
    ax3 = axes('units','normalized','position',[.15 .09 .8 2*.8/3]);
    image(ax3,255*PAL_Scale0to1((flipud(LLspace))));
    set(ax3,'xtick',[],'ytick',[]);
    drawnow
    pause
    clf;
end
